<?php

namespace DZ\Server\Service;

use XF\Service\AbstractService;

class ServerInfo extends AbstractService
{
    protected int $timeout = 5;

    protected function runCommand(string $command): string
    {
        if (!function_exists('shell_exec') || !is_callable('shell_exec')) {
            return '';
        }

        return trim((string)@shell_exec($command));
    }

    protected function calculatePct(float|int $used, float|int $total): float
    {
        return $total > 0 ? round(($used / $total) * 100, 2) : 0;
    }

    protected function getBarClass(float $pct): string
    {
        if ($pct >= 85) {
            return 'is-red';
        }
        if ($pct >= 50) {
            return 'is-orange';
        }
        return 'is-green';
    }

    protected function formatUptime(int $seconds): string
    {
        if ($seconds <= 0) {
            return '0m';
        }

        $days = intdiv($seconds, 86400);
        $seconds %= 86400;
        $hours = intdiv($seconds, 3600);
        $seconds %= 3600;
        $minutes = intdiv($seconds, 60);

        $parts = [];
        if ($days > 0) {
            $parts[] = $days . 'd';
        }
        if ($hours > 0) {
            $parts[] = $hours . 'h';
        }
        if ($minutes > 0 || !$parts) {
            $parts[] = $minutes . 'm';
        }

        return implode(' ', $parts);
    }

    /**
     * Glavni ulaz – bira implementaciju po OS-u i kešira rezultat.
     */
    public function getServerData(): array
    {
        $cache = $this->app->cache();
        $cacheKey = 'dz_server_info_data';

        if ($data = $cache?->fetch($cacheKey)) {
            return $data;
        }

        $family = PHP_OS_FAMILY;

        if ($family === 'Windows') {
            $data = $this->getWindowsStats();
        } elseif ($family === 'BSD') {
            $data = $this->getFreeBSDStats();
        } else {
            $data = $this->getLinuxStats();
        }

        $cache?->save($cacheKey, $data, 60);

        return $data;
    }

    /**
     * Linux implementacija (original koju si već koristio).
     */
    protected function getLinuxStats(): array
    {
        $totalRam = 0;
        $usedRam = 0;
        $freeRam = 0;
        $availableRam = 0;

        $totalSwap = 0;
        $usedSwap = 0;
        $freeSwap = 0;

        $cpuUsagePct = 0.0;
        $cpuLoad1Min = 0.0;
        $cpuLoad5Min = 0.0;
        $cpuLoad15Min = 0.0;

        $cpuIoWait = 0.0;
        $cpuCores = 0;
        $cpuModel = 'N/A';
        $cpuTemp = 0.0;

        $totalDisk = 0;
        $usedDisk = 0;
        $freeDisk = 0;
        $diskUsagePct = 0;

        $establishedConnections = 0;
        $totalConnections = 0;
        $activeTcp = 0;
        $activeUdp = 0;

        $netRxMB = 0.0;
        $netTxMB = 0.0;

        $hostname = 'N/A';
        $ipAddress = 'N/A';
        $uptimeFormatted = 'N/A';

        // RAM
        $memLine = $this->runCommand("free -m | awk 'NR==2 {print \$2, \$3, \$4, \$7}'");
        $memParts = preg_split('/\s+/', trim($memLine));
        $totalRam     = (int)($memParts[0] ?? 0);
        $usedRam      = (int)($memParts[1] ?? 0);
        $freeRam      = (int)($memParts[2] ?? 0);
        $availableRam = (int)($memParts[3] ?? 0);

        // SWAP
        $swapLine = $this->runCommand("free -m | awk 'NR==3 {print \$2, \$3, \$4}'");
        $swapParts = preg_split('/\s+/', trim($swapLine));
        $totalSwap = (int)($swapParts[0] ?? 0);
        $usedSwap  = (int)($swapParts[1] ?? 0);
        $freeSwap  = (int)($swapParts[2] ?? 0);

        // CPU usage
        $cpuUsagePct = (float)$this->runCommand("top -bn1 | grep 'Cpu(s)' | awk '{print 100 - \$8}'");

        // IO wait
        $cpuIoWait = (float)$this->runCommand("top -bn1 | grep 'Cpu(s)' | awk '{print \$10}'");

        // LOAD AVERAGE
        $loadLine = $this->runCommand("cat /proc/loadavg | awk '{print \$1, \$2, \$3}'");
        $loadParts = preg_split('/\s+/', trim($loadLine));
        $cpuLoad1Min  = (float)($loadParts[0] ?? 0);
        $cpuLoad5Min  = (float)($loadParts[1] ?? 0);
        $cpuLoad15Min = (float)($loadParts[2] ?? 0);

        // UPTIME
        $uptimeSec = (int)$this->runCommand("awk '{print int(\$1)}' /proc/uptime");
        $uptimeFormatted = $this->formatUptime($uptimeSec);

        // CPU cores
        $cpuCores = (int)$this->runCommand('nproc');

        // CPU model
        $cpuModelLine = $this->runCommand("grep -m1 'model name' /proc/cpuinfo | cut -d: -f2");
        $cpuModel = $cpuModelLine !== '' ? trim($cpuModelLine) : 'N/A';

        // CPU temperatura
        $cpuTempRaw = $this->runCommand("sensors 2>/dev/null | awk '/Core 0/ {print \$3}' | head -n1 | sed 's/+//;s/°C//'");
        if ($cpuTempRaw === '') {
            $cpuTempRaw = $this->runCommand("awk 'NR==1 {print \$1/1000}' /sys/class/thermal/thermal_zone0/temp 2>/dev/null");
        }
        $cpuTemp = (float)$cpuTempRaw;

        // DISK — root filesystem
        $df = $this->runCommand("df -m / | awk 'END {print \$2, \$3, \$4}'");
        $diskParts = preg_split('/\s+/', trim($df));
        $totalDisk = (int)($diskParts[0] ?? 0);
        $usedDisk  = (int)($diskParts[1] ?? 0);
        $freeDisk  = (int)($diskParts[2] ?? 0);
        $diskUsagePct = $this->calculatePct($usedDisk, $totalDisk);

        // CONNECTIONS
        $establishedConnections = (int)$this->runCommand("ss -ant | grep -c ESTAB || echo 0");
        $totalConnections       = (int)$this->runCommand("ss -ant | tail -n +2 | wc -l || echo 0");
        $activeTcp              = $totalConnections;
        $activeUdp              = (int)$this->runCommand("ss -anu | tail -n +2 | wc -l || echo 0");

        // NETWORK (MB)
        $netLine = $this->runCommand("cat /proc/net/dev | awk 'NR>2 {gsub(\":\", \"\", \$1); if (\$1 != \"lo\") {rx+=\$2; tx+=\$10}} END {print rx, tx}'");
        $netParts = preg_split('/\s+/', trim($netLine));
        $netRxBytes = (float)($netParts[0] ?? 0);
        $netTxBytes = (float)($netParts[1] ?? 0);

        $netRxMB = round($netRxBytes / 1024 / 1024, 2);
        $netTxMB = round($netTxBytes / 1024 / 1024, 2);

        // Hostname & IP
        $hostname = $this->runCommand('hostname');
        $ipAddress = $this->runCommand("hostname -I | awk '{print \$1}'");

        // PERCENTAGES
        $ramUsagePct  = $this->calculatePct($usedRam, $totalRam);
        $swapUsagePct = $this->calculatePct($usedSwap, $totalSwap);

        $forumRamMB  = round(memory_get_usage(true) / 1024 / 1024, 2);
        $forumRamPct = $this->calculatePct($forumRamMB, $totalRam);

        $phpVersion = PHP_VERSION;

        // CLASSES
        $ramClass    = $this->getBarClass($ramUsagePct);
        $cpuClass    = $this->getBarClass($cpuUsagePct);
        $diskClass   = $this->getBarClass($diskUsagePct);
        $forumClass  = $this->getBarClass($forumRamPct);
        $swapClass   = $this->getBarClass($swapUsagePct);

        return [
            'ramUsagePct'   => $ramUsagePct,
            'cpuUsagePct'   => $cpuUsagePct,
            'diskUsagePct'  => $diskUsagePct,
            'forumRamPct'   => $forumRamPct,
            'swapUsagePct'  => $swapUsagePct,

            'ramClass'      => $ramClass,
            'cpuClass'      => $cpuClass,
            'diskClass'     => $diskClass,
            'forumRamClass' => $forumClass,
            'swapClass'     => $swapClass,

            'totalRam'      => "{$totalRam} MB",
            'usedRam'       => "{$usedRam} MB",
            'freeRam'       => "{$freeRam} MB",
            'availableRam'  => "{$availableRam} MB",

            'swapTotal'     => "{$totalSwap} MB",
            'swapUsed'      => "{$usedSwap} MB",
            'swapFree'      => "{$freeSwap} MB",

            'totalDisk'     => $totalDisk,
            'usedDisk'      => $usedDisk,
            'freeDisk'      => $freeDisk,

            'establishedConnections' => $establishedConnections,
            'totalConnections'       => $totalConnections,
            'activeTcp'              => $activeTcp,
            'activeUdp'              => $activeUdp,

            'netRxMB'       => $netRxMB,
            'netTxMB'       => $netTxMB,

            'hostname'      => $hostname,
            'ipAddress'     => $ipAddress,
            'phpVersion'    => $phpVersion,

            'phpLoad'       => "{$forumRamMB} MB",

            'loadTime'      => $cpuLoad1Min,
            'cpuLoad1'      => $cpuLoad1Min,
            'cpuLoad5'      => $cpuLoad5Min,
            'cpuLoad15'     => $cpuLoad15Min,
            'cpuCores'      => $cpuCores,
            'cpuModel'      => $cpuModel,
            'cpuIoWait'     => $cpuIoWait,
            'cpuTemp'       => $cpuTemp,

            'uptime'        => $uptimeFormatted,
        ];
    }

    /**
     * FreeBSD implementacija – koristi sysctl, vmstat, df, netstat, sockstat...
     */
    protected function getFreeBSDStats(): array
    {
        // RAM
        $mem = $this->runCommand("sysctl hw.physmem hw.pagesize vm.stats.vm.v_page_count vm.stats.vm.v_free_count");
        preg_match_all('/\d+/', $mem, $m);

        $physMem   = (int)($m[0][0] ?? 0);
        $pageSize  = (int)($m[0][1] ?? 4096);
        $pageCount = (int)($m[0][2] ?? 0);
        $freePages = (int)($m[0][3] ?? 0);

        $totalRam = round($physMem / 1024 / 1024);
        $freeRam  = round(($freePages * $pageSize) / 1024 / 1024);
        $usedRam  = $totalRam - $freeRam;
        $availableRam = $freeRam;

        // Swap
        $swap = $this->runCommand("swapinfo -k | awk 'NR==2 {print \$2, \$3, \$4}'");
        $s = preg_split('/\s+/', trim($swap));
        $totalSwap = isset($s[0]) ? round($s[0] / 1024) : 0;
        $usedSwap  = isset($s[1]) ? round($s[1] / 1024) : 0;
        $freeSwap  = isset($s[2]) ? round($s[2] / 1024) : 0;

        // CPU load 1/5/15
        $load = $this->runCommand("sysctl -n vm.loadavg");
        preg_match_all('/\d+\.\d+/', $load, $l);
        $cpuLoad1 = (float)($l[0][0] ?? 0);
        $cpuLoad5 = (float)($l[0][1] ?? 0);
        $cpuLoad15 = (float)($l[0][2] ?? 0);

        // CPU cores / model
        $cpuCores = (int)$this->runCommand("sysctl -n hw.ncpu");
        $cpuModel = trim($this->runCommand("sysctl -n hw.model"));

        // CPU usage (approx) – koristi top
        $cpuTop = $this->runCommand("top -b -n 1 | grep 'CPU:'");
        preg_match('/(\d+\.\d+)%\sidle/', $cpuTop, $idle);
        $idlePct = (float)($idle[1] ?? 0);
        $cpuUsagePct = round(100 - $idlePct, 2);

        // IO wait (best effort)
        preg_match('/(\d+\.\d+)%\siowait/', $cpuTop, $io);
        $cpuIoWait = (float)($io[1] ?? 0);

        // Temperature (ako postoji coretemp)
        $temp = $this->runCommand("sysctl -n dev.cpu.0.temperature 2>/dev/null | sed 's/C//' | sed 's/+//' | sed 's/°//' | sed 's/ //g'");
        $cpuTemp = 0.0;
        if ($temp !== '' && is_numeric($temp)) {
            $cpuTemp = (float)$temp;
        }

        // Disk usage (root /)
        $df = $this->runCommand("df -m / | awk 'NR==2 {print \$2, \$3, \$4}'");
        $d = preg_split('/\s+/', trim($df));
        $totalDisk = (int)($d[0] ?? 0);
        $usedDisk  = (int)($d[1] ?? 0);
        $freeDisk  = (int)($d[2] ?? 0);
        $diskUsagePct = $this->calculatePct($usedDisk, $totalDisk);

        // Network RX/TX (svi interface-i)
        $net = $this->runCommand("netstat -ib | awk 'NR>1 {rx+=\$7; tx+=\$10} END {print rx, tx}'");
        $n = preg_split('/\s+/', trim($net));
        $rxMB = round(((float)($n[0] ?? 0)) / 1024 / 1024, 2);
        $txMB = round(((float)($n[1] ?? 0)) / 1024 / 1024, 2);

        // Connections (sockstat)
        $established = (int)$this->runCommand("sockstat -4t | grep -c ESTABLISHED 2>/dev/null || echo 0");
        $totalCon    = (int)$this->runCommand("sockstat -4 | wc -l 2>/dev/null || echo 0");
        $tcpCon      = (int)$this->runCommand("sockstat -4t | wc -l 2>/dev/null || echo 0");
        $udpCon      = (int)$this->runCommand("sockstat -4u | wc -l 2>/dev/null || echo 0");

        // Uptime
        $boot = (int)$this->runCommand("sysctl -n kern.boottime | sed 's/.*sec = \\([0-9]*\\).*/\\1/'");
        $uptimeFormatted = $this->formatUptime(time() - $boot);

        // Percentages
        $ramUsagePct  = $this->calculatePct($usedRam, $totalRam);
        $swapUsagePct = $this->calculatePct($usedSwap, $totalSwap);

        $forumRamMB  = round(memory_get_usage(true) / 1024 / 1024, 2);
        $forumRamPct = $this->calculatePct($forumRamMB, $totalRam);

        $hostname = trim($this->runCommand('hostname'));
        // FreeBSD nema hostname -I, pa vadimo inet (osim 127.0.0.1)
        $ipAddress = trim($this->runCommand("ifconfig | awk '/inet / && \$2 != \"127.0.0.1\" {print \$2; exit}'"));
        $phpVersion = PHP_VERSION;

        return [
            'ramUsagePct'   => $ramUsagePct,
            'cpuUsagePct'   => $cpuUsagePct,
            'diskUsagePct'  => $diskUsagePct,
            'forumRamPct'   => $forumRamPct,
            'swapUsagePct'  => $swapUsagePct,

            'ramClass'      => $this->getBarClass($ramUsagePct),
            'cpuClass'      => $this->getBarClass($cpuUsagePct),
            'diskClass'     => $this->getBarClass($diskUsagePct),
            'forumRamClass' => $this->getBarClass($forumRamPct),
            'swapClass'     => $this->getBarClass($swapUsagePct),

            'totalRam'      => "{$totalRam} MB",
            'usedRam'       => "{$usedRam} MB",
            'freeRam'       => "{$freeRam} MB",
            'availableRam'  => "{$availableRam} MB",

            'swapTotal'     => "{$totalSwap} MB",
            'swapUsed'      => "{$usedSwap} MB",
            'swapFree'      => "{$freeSwap} MB",

            'totalDisk'     => $totalDisk,
            'usedDisk'      => $usedDisk,
            'freeDisk'      => $freeDisk,

            'establishedConnections' => $established,
            'totalConnections'       => $totalCon,
            'activeTcp'              => $tcpCon,
            'activeUdp'              => $udpCon,

            'netRxMB'       => $rxMB,
            'netTxMB'       => $txMB,

            'hostname'      => $hostname,
            'ipAddress'     => $ipAddress,
            'phpVersion'    => $phpVersion,

            'phpLoad'       => "{$forumRamMB} MB",

            'loadTime'      => $cpuLoad1,
            'cpuLoad1'      => $cpuLoad1,
            'cpuLoad5'      => $cpuLoad5,
            'cpuLoad15'     => $cpuLoad15,
            'cpuCores'      => $cpuCores,
            'cpuModel'      => $cpuModel,
            'cpuIoWait'     => $cpuIoWait,
            'cpuTemp'       => $cpuTemp,

            'uptime'        => $uptimeFormatted,
        ];
    }

    /**
     * Windows: pokušaj prvo COM/WMI, pa PowerShell, pa fallback na demo.
     */
    protected function getWindowsStats(): array
    {
        // 1) COM/WMI ako je dostupno
        if (class_exists('\\COM')) {
            try {
                $data = $this->collectWindowsStatsViaCom();
                if (is_array($data) && $data) {
                    return $data;
                }
            } catch (\Throwable $e) {
                // ignore, fallback
            }
        }

        // 2) PowerShell preko shell_exec
        $data = $this->collectWindowsStatsViaPowerShell();
        if (is_array($data) && $data) {
            return $data;
        }

        // 3) Fallback demo
        return $this->getWindowsFallbackStats();
    }

    /**
     * Windows – COM / WMI implementacija.
     */
    protected function collectWindowsStatsViaCom(): ?array
    {
        try {
            $wmi = new \COM("winmgmts:{impersonationLevel=impersonate}//./root/cimv2");

            // OS info (RAM, swap, uptime)
            $osSet = $wmi->ExecQuery("SELECT TotalVisibleMemorySize,FreePhysicalMemory,TotalVirtualMemorySize,FreeVirtualMemory,LastBootUpTime FROM Win32_OperatingSystem");
            $osInfo = null;
            foreach ($osSet as $item) {
                $osInfo = $item;
                break;
            }
            if (!$osInfo) {
                return null;
            }

            $totalRam = (int)round(($osInfo->TotalVisibleMemorySize ?? 0) / 1024);
            $freeRam  = (int)round(($osInfo->FreePhysicalMemory ?? 0) / 1024);
            $usedRam  = max(0, $totalRam - $freeRam);
            $availableRam = $freeRam;

            $totalVirtualKB = (int)($osInfo->TotalVirtualMemorySize ?? 0);
            $freeVirtualKB  = (int)($osInfo->FreeVirtualMemory ?? 0);
            $swapTotalKB    = max(0, $totalVirtualKB - (int)$osInfo->TotalVisibleMemorySize);
            $swapFreeKB     = max(0, $freeVirtualKB - (int)$osInfo->FreePhysicalMemory);

            $totalSwap = (int)round($swapTotalKB / 1024);
            $freeSwap  = (int)round($swapFreeKB / 1024);
            $usedSwap  = max(0, $totalSwap - $freeSwap);

            // CPU
            $cpuSet = $wmi->ExecQuery("SELECT LoadPercentage, Name, NumberOfLogicalProcessors FROM Win32_Processor");
            $cpuUsagePct = 0.0;
            $cpuModel = 'Windows CPU';
            $cpuCores = 0;

            foreach ($cpuSet as $cpu) {
                $cpuUsagePct = (float)($cpu->LoadPercentage ?? 0);
                $cpuModel = (string)($cpu->Name ?? 'Windows CPU');
                $cpuCores = (int)($cpu->NumberOfLogicalProcessors ?? 0);
                break;
            }

            $cpuLoad1  = round($cpuUsagePct * max($cpuCores, 1) / 100, 2);
            $cpuLoad5  = $cpuLoad1;
            $cpuLoad15 = $cpuLoad1;

            // IO wait – nema direktno, ostavimo 0
            $cpuIoWait = 0.0;

            // CPU temp – ako postoji (ostavljamo 0, ovo je dosta vendor-specifično)
            $cpuTemp = 0.0;

            // Disk usage – sumiramo sve lokalne diskove
            $diskSet = $wmi->ExecQuery("SELECT Size, FreeSpace FROM Win32_LogicalDisk WHERE DriveType=3");
            $totalDisk = 0;
            $freeDisk  = 0;

            foreach ($diskSet as $disk) {
                $size = (float)($disk->Size ?? 0);
                $free = (float)($disk->FreeSpace ?? 0);
                $totalDisk += (int)round($size / 1024 / 1024);
                $freeDisk  += (int)round($free / 1024 / 1024);
            }
            $usedDisk = max(0, $totalDisk - $freeDisk);
            $diskUsagePct = $this->calculatePct($usedDisk, $totalDisk);

            // Connections – preko netstat
            $establishedConnections = (int)$this->runCommand('netstat -an | find "ESTABLISHED" /c');
            $totalConnections       = (int)$this->runCommand('netstat -an | find ":" /c');
            $activeTcp              = $establishedConnections;
            $activeUdp              = (int)$this->runCommand('netstat -an | find "UDP" /c');

            // Network – nije trivijalno kumulativno, ostavljamo 0
            $netRxMB = 0.0;
            $netTxMB = 0.0;

            // Hostname & IP
            $hostname = php_uname('n');
            $ipAddress = gethostbyname($hostname);

            // Uptime
            $uptimeFormatted = 'N/A';
            $bootRaw = (string)($osInfo->LastBootUpTime ?? '');
            if ($bootRaw !== '') {
                $bootStr = substr($bootRaw, 0, 14);
                $dt = \DateTime::createFromFormat('YmdHis', $bootStr, new \DateTimeZone('UTC'));
                if ($dt instanceof \DateTime) {
                    $uptimeSec = time() - $dt->getTimestamp();
                    $uptimeFormatted = $this->formatUptime($uptimeSec);
                }
            }

            // Percentages
            $ramUsagePct  = $this->calculatePct($usedRam, $totalRam);
            $swapUsagePct = $this->calculatePct($usedSwap, $totalSwap);
            $forumRamMB   = round(memory_get_usage(true) / 1024 / 1024, 2);
            $forumRamPct  = $this->calculatePct($forumRamMB, $totalRam);

            $phpVersion = PHP_VERSION;

            return [
                'ramUsagePct'   => $ramUsagePct,
                'cpuUsagePct'   => $cpuUsagePct,
                'diskUsagePct'  => $diskUsagePct,
                'forumRamPct'   => $forumRamPct,
                'swapUsagePct'  => $swapUsagePct,

                'ramClass'      => $this->getBarClass($ramUsagePct),
                'cpuClass'      => $this->getBarClass($cpuUsagePct),
                'diskClass'     => $this->getBarClass($diskUsagePct),
                'forumRamClass' => $this->getBarClass($forumRamPct),
                'swapClass'     => $this->getBarClass($swapUsagePct),

                'totalRam'      => "{$totalRam} MB",
                'usedRam'       => "{$usedRam} MB",
                'freeRam'       => "{$freeRam} MB",
                'availableRam'  => "{$availableRam} MB",

                'swapTotal'     => "{$totalSwap} MB",
                'swapUsed'      => "{$usedSwap} MB",
                'swapFree'      => "{$freeSwap} MB",

                'totalDisk'     => $totalDisk,
                'usedDisk'      => $usedDisk,
                'freeDisk'      => $freeDisk,

                'establishedConnections' => $establishedConnections,
                'totalConnections'       => $totalConnections,
                'activeTcp'              => $activeTcp,
                'activeUdp'              => $activeUdp,

                'netRxMB'       => $netRxMB,
                'netTxMB'       => $netTxMB,

                'hostname'      => $hostname,
                'ipAddress'     => $ipAddress,
                'phpVersion'    => $phpVersion,

                'phpLoad'       => "{$forumRamMB} MB",

                'loadTime'      => $cpuLoad1,
                'cpuLoad1'      => $cpuLoad1,
                'cpuLoad5'      => $cpuLoad5,
                'cpuLoad15'     => $cpuLoad15,
                'cpuCores'      => $cpuCores,
                'cpuModel'      => $cpuModel,
                'cpuIoWait'     => $cpuIoWait,
                'cpuTemp'       => $cpuTemp,

                'uptime'        => $uptimeFormatted,
            ];
        } catch (\Throwable $e) {
            return null;
        }
    }

    /**
     * Windows – PowerShell implementacija (Get-CimInstance, Get-NetAdapterStatistics...)
     */
    protected function collectWindowsStatsViaPowerShell(): ?array
    {
        // OS / RAM / swap / uptime
        $osJson = $this->runCommand(
            'powershell -NoProfile -Command "Get-CimInstance Win32_OperatingSystem | ' .
            'Select-Object TotalVisibleMemorySize,FreePhysicalMemory,TotalVirtualMemorySize,FreeVirtualMemory,LastBootUpTime | ' .
            'ConvertTo-Json -Compress"'
        );
        if ($osJson === '') {
            return null;
        }

        $osInfo = @json_decode($osJson, true);
        if (!is_array($osInfo)) {
            return null;
        }

        $totalRam = (int)round(($osInfo['TotalVisibleMemorySize'] ?? 0) / 1024);
        $freeRam  = (int)round(($osInfo['FreePhysicalMemory'] ?? 0) / 1024);
        $usedRam  = max(0, $totalRam - $freeRam);
        $availableRam = $freeRam;

        $totalVirtualKB = (int)($osInfo['TotalVirtualMemorySize'] ?? 0);
        $freeVirtualKB  = (int)($osInfo['FreeVirtualMemory'] ?? 0);
        $swapTotalKB    = max(0, $totalVirtualKB - (int)($osInfo['TotalVisibleMemorySize'] ?? 0));
        $swapFreeKB     = max(0, $freeVirtualKB - (int)($osInfo['FreePhysicalMemory'] ?? 0));

        $totalSwap = (int)round($swapTotalKB / 1024);
        $freeSwap  = (int)round($swapFreeKB / 1024);
        $usedSwap  = max(0, $totalSwap - $freeSwap);

        // CPU
        $cpuUsageStr = $this->runCommand(
            'powershell -NoProfile -Command "(Get-CimInstance Win32_Processor | ' .
            'Measure-Object -Property LoadPercentage -Average).Average"'
        );
        $cpuUsagePct = (float)trim($cpuUsageStr);

        $cpuModel = trim($this->runCommand(
            'powershell -NoProfile -Command "(Get-CimInstance Win32_Processor | ' .
            'Select-Object -First 1 -ExpandProperty Name)"'
        ));
        if ($cpuModel === '') {
            $cpuModel = 'Windows CPU';
        }

        $cpuCoresStr = $this->runCommand(
            'powershell -NoProfile -Command "(Get-CimInstance Win32_Processor | ' .
            'Select-Object -First 1 -ExpandProperty NumberOfLogicalProcessors)"'
        );
        $cpuCores = (int)trim($cpuCoresStr);
        $cpuCores = max($cpuCores, 1);

        $cpuLoad1  = round($cpuUsagePct * $cpuCores / 100, 2);
        $cpuLoad5  = $cpuLoad1;
        $cpuLoad15 = $cpuLoad1;
        $cpuIoWait = 0.0;
        $cpuTemp   = 0.0;

        // Disk
        $diskJson = $this->runCommand(
            'powershell -NoProfile -Command "Get-CimInstance Win32_LogicalDisk -Filter \"DriveType=3\" | ' .
            'Select-Object Size,FreeSpace | ConvertTo-Json -Compress"'
        );
        $diskInfo = @json_decode($diskJson, true);

        $totalDisk = 0;
        $freeDisk  = 0;

        if (is_array($diskInfo)) {
            // može biti single obj ili array
            if (isset($diskInfo['Size'])) {
                $diskInfo = [$diskInfo];
            }
            foreach ($diskInfo as $disk) {
                $size = (float)($disk['Size'] ?? 0);
                $free = (float)($disk['FreeSpace'] ?? 0);
                $totalDisk += (int)round($size / 1024 / 1024);
                $freeDisk  += (int)round($free / 1024 / 1024);
            }
        }

        $usedDisk = max(0, $totalDisk - $freeDisk);
        $diskUsagePct = $this->calculatePct($usedDisk, $totalDisk);

        // Network RX/TX (Get-NetAdapterStatistics)
        $netJson = $this->runCommand(
            'powershell -NoProfile -Command "Get-NetAdapterStatistics | ' .
            'Select-Object ReceivedBytes,SentBytes | ConvertTo-Json -Compress"'
        );
        $netInfo = @json_decode($netJson, true);
        $rxBytesTotal = 0.0;
        $txBytesTotal = 0.0;

        if (is_array($netInfo)) {
            if (isset($netInfo['ReceivedBytes'])) {
                $netInfo = [$netInfo];
            }
            foreach ($netInfo as $ni) {
                $rxBytesTotal += (float)($ni['ReceivedBytes'] ?? 0);
                $txBytesTotal += (float)($ni['SentBytes'] ?? 0);
            }
        }

        $netRxMB = round($rxBytesTotal / 1024 / 1024, 2);
        $netTxMB = round($txBytesTotal / 1024 / 1024, 2);

        // Connections (Get-NetTCPConnection)
        $establishedStr = $this->runCommand(
            'powershell -NoProfile -Command "(Get-NetTCPConnection -State Established | Measure-Object).Count"'
        );
        $totalConStr = $this->runCommand(
            'powershell -NoProfile -Command "(Get-NetTCPConnection | Measure-Object).Count"'
        );
        $udpConStr = $this->runCommand(
            'powershell -NoProfile -Command "(Get-NetUDPEndpoint | Measure-Object).Count"'
        );

        $establishedConnections = (int)trim($establishedStr);
        $totalConnections       = (int)trim($totalConStr);
        $activeTcp              = $establishedConnections;
        $activeUdp              = (int)trim($udpConStr);

        // Uptime
        $uptimeFormatted = 'N/A';
        if (!empty($osInfo['LastBootUpTime'])) {
            $bootRaw = $osInfo['LastBootUpTime'];
            $bootStr = substr($bootRaw, 0, 14);
            $dt = \DateTime::createFromFormat('YmdHis', $bootStr, new \DateTimeZone('UTC'));
            if ($dt instanceof \DateTime) {
                $uptimeSec = time() - $dt->getTimestamp();
                $uptimeFormatted = $this->formatUptime($uptimeSec);
            }
        }

        // Percentages
        $ramUsagePct  = $this->calculatePct($usedRam, $totalRam);
        $swapUsagePct = $this->calculatePct($usedSwap, $totalSwap);
        $forumRamMB   = round(memory_get_usage(true) / 1024 / 1024, 2);
        $forumRamPct  = $this->calculatePct($forumRamMB, $totalRam);

        $hostname = php_uname('n');
        $ipAddress = gethostbyname($hostname);
        $phpVersion = PHP_VERSION;

        return [
            'ramUsagePct'   => $ramUsagePct,
            'cpuUsagePct'   => $cpuUsagePct,
            'diskUsagePct'  => $diskUsagePct,
            'forumRamPct'   => $forumRamPct,
            'swapUsagePct'  => $swapUsagePct,

            'ramClass'      => $this->getBarClass($ramUsagePct),
            'cpuClass'      => $this->getBarClass($cpuUsagePct),
            'diskClass'     => $this->getBarClass($diskUsagePct),
            'forumRamClass' => $this->getBarClass($forumRamPct),
            'swapClass'     => $this->getBarClass($swapUsagePct),

            'totalRam'      => "{$totalRam} MB",
            'usedRam'       => "{$usedRam} MB",
            'freeRam'       => "{$freeRam} MB",
            'availableRam'  => "{$availableRam} MB",

            'swapTotal'     => "{$totalSwap} MB",
            'swapUsed'      => "{$usedSwap} MB",
            'swapFree'      => "{$freeSwap} MB",

            'totalDisk'     => $totalDisk,
            'usedDisk'      => $usedDisk,
            'freeDisk'      => $freeDisk,

            'establishedConnections' => $establishedConnections,
            'totalConnections'       => $totalConnections,
            'activeTcp'              => $activeTcp,
            'activeUdp'              => $activeUdp,

            'netRxMB'       => $netRxMB,
            'netTxMB'       => $netTxMB,

            'hostname'      => $hostname,
            'ipAddress'     => $ipAddress,
            'phpVersion'    => $phpVersion,

            'phpLoad'       => "{$forumRamMB} MB",

            'loadTime'      => $cpuLoad1,
            'cpuLoad1'      => $cpuLoad1,
            'cpuLoad5'      => $cpuLoad5,
            'cpuLoad15'     => $cpuLoad15,
            'cpuCores'      => $cpuCores,
            'cpuModel'      => $cpuModel,
            'cpuIoWait'     => $cpuIoWait,
            'cpuTemp'       => $cpuTemp,

            'uptime'        => $uptimeFormatted,
        ];
    }

    /**
     * Windows fallback demo – ako COM i PowerShell ne rade.
     */
    protected function getWindowsFallbackStats(): array
    {
        $totalRam = 8192;
        $usedRam  = 4096;
        $freeRam  = $totalRam - $usedRam;
        $availableRam = $freeRam;

        $totalSwap = 2048;
        $usedSwap  = 256;
        $freeSwap  = $totalSwap - $usedSwap;

        $cpuUsagePct = 25.0;
        $cpuLoad1 = 1.0;
        $cpuLoad5 = 0.8;
        $cpuLoad15 = 0.5;
        $cpuCores = 8;
        $cpuModel = 'Windows CPU (fallback)';
        $cpuIoWait = 0.0;
        $cpuTemp = 50.0;

        $totalDisk = 256000;
        $usedDisk  = 128000;
        $freeDisk  = $totalDisk - $usedDisk;
        $diskUsagePct = $this->calculatePct($usedDisk, $totalDisk);

        $establishedConnections = 10;
        $totalConnections       = 20;
        $activeTcp              = 8;
        $activeUdp              = 4;

        $netRxMB = 100.0;
        $netTxMB = 50.0;

        $hostname = php_uname('n');
        $ipAddress = gethostbyname($hostname);

        $ramUsagePct  = $this->calculatePct($usedRam, $totalRam);
        $swapUsagePct = $this->calculatePct($usedSwap, $totalSwap);
        $forumRamMB   = round(memory_get_usage(true) / 1024 / 1024, 2);
        $forumRamPct  = $this->calculatePct($forumRamMB, $totalRam);

        $phpVersion = PHP_VERSION;

        $uptimeFormatted = '1d 2h 30m';

        return [
            'ramUsagePct'   => $ramUsagePct,
            'cpuUsagePct'   => $cpuUsagePct,
            'diskUsagePct'  => $diskUsagePct,
            'forumRamPct'   => $forumRamPct,
            'swapUsagePct'  => $swapUsagePct,

            'ramClass'      => $this->getBarClass($ramUsagePct),
            'cpuClass'      => $this->getBarClass($cpuUsagePct),
            'diskClass'     => $this->getBarClass($diskUsagePct),
            'forumRamClass' => $this->getBarClass($forumRamPct),
            'swapClass'     => $this->getBarClass($swapUsagePct),

            'totalRam'      => "{$totalRam} MB",
            'usedRam'       => "{$usedRam} MB",
            'freeRam'       => "{$freeRam} MB",
            'availableRam'  => "{$availableRam} MB",

            'swapTotal'     => "{$totalSwap} MB",
            'swapUsed'      => "{$usedSwap} MB",
            'swapFree'      => "{$freeSwap} MB",

            'totalDisk'     => $totalDisk,
            'usedDisk'      => $usedDisk,
            'freeDisk'      => $freeDisk,

            'establishedConnections' => $establishedConnections,
            'totalConnections'       => $totalConnections,
            'activeTcp'              => $activeTcp,
            'activeUdp'              => $activeUdp,

            'netRxMB'       => $netRxMB,
            'netTxMB'       => $netTxMB,

            'hostname'      => $hostname,
            'ipAddress'     => $ipAddress,
            'phpVersion'    => $phpVersion,

            'phpLoad'       => "{$forumRamMB} MB",

            'loadTime'      => $cpuLoad1,
            'cpuLoad1'      => $cpuLoad1,
            'cpuLoad5'      => $cpuLoad5,
            'cpuLoad15'     => $cpuLoad15,
            'cpuCores'      => $cpuCores,
            'cpuModel'      => $cpuModel,
            'cpuIoWait'     => $cpuIoWait,
            'cpuTemp'       => $cpuTemp,

            'uptime'        => $uptimeFormatted,
        ];
    }
}
