<?php
/**
 * Language Pack Repository Generator
 * 
 * This script scans folders and generates languages.json automatically.
 * 
 * Usage: 
 * 1. Upload XML files to appropriate folders
 * 2. Run this script: https://yourdomain.com/repo/language_installer/generate.php
 * 3. languages.json will be regenerated
 *
 * Folder structure:
 * - Language folders: Use language code (e.g., hr-HR/, ru-RU/)
 * - Addon folders: Use dash instead of slash (e.g., MMO-Hide/ for MMO/Hide addon)
 *
 * Security: Add ?key=YOUR_SECRET_KEY to prevent unauthorized access
 */

// Simple security - change this key!
$secretKey = 'dz2026';

if (!isset($_GET['key']) || $_GET['key'] !== $secretKey) {
    die('Access denied. Use ?key=YOUR_SECRET_KEY');
}

$baseDir = __DIR__;
$languages = [];
$addons = [];

// Known addon titles (folder name => display title)
$addonTitles = [
    'XF' => 'XenForo Core',
    'XFRM' => 'XenForo Resource Manager',
    'XFMG' => 'XenForo Media Gallery',
    'XFES' => 'XenForo Enhanced Search',
    'XFPN' => 'XenForo Push Notifications',
    'DBTech-Security' => 'DBTech Security',
    'MMO-Hide' => 'MMO Hide',
];

/**
 * Convert folder name to XenForo addon ID
 * MMO-Hide -> MMO/Hide
 * XFMG -> XFMG (unchanged if no dash or known addon)
 */
function folderToAddonId($folderName) {
    // Known addons that DON'T need conversion (no slash in real addon ID)
    $simpleAddons = ['XFMG', 'XFRM', 'XFES', 'XFPN', 'XF'];
    
    if (in_array($folderName, $simpleAddons)) {
        return $folderName;
    }
    
    // Check if folder name contains a dash that should become a slash
    // Pattern: Vendor-AddonName (e.g., MMO-Hide, DZ-AutoDelete)
    if (preg_match('/^([A-Za-z0-9]+)-([A-Za-z0-9]+)$/', $folderName, $matches)) {
        return $matches[1] . '/' . $matches[2];
    }
    
    return $folderName;
}

/**
 * Extract metadata from XenForo language XML file
 */
function extractXmlMetadata($filePath) {
    $content = file_get_contents($filePath);
    
    $metadata = [
        'title' => null,
        'code' => null,
        'version' => null,
        'addon_id' => null,
    ];
    
    // Extract language_code
    if (preg_match('/language_code=["\']([^"\']+)["\']/', $content, $matches)) {
        $metadata['code'] = $matches[1];
    }
    
    // Extract title
    if (preg_match('/title=["\']([^"\']+)["\']/', $content, $matches)) {
        $metadata['title'] = $matches[1];
    }
    
    // Extract base_version_id and convert to readable version
    if (preg_match('/base_version_id=["\'](\d+)["\']/', $content, $matches)) {
        $versionId = (int)$matches[1];
        $major = floor($versionId / 1000000);
        $minor = floor(($versionId % 1000000) / 10000);
        $patch = floor(($versionId % 10000) / 100);
        $metadata['version'] = $major . '.' . $minor . '.' . $patch;
    }
    
    // Extract addon_id from XML (if present)
    if (preg_match('/addon_id=["\']([^"\']+)["\']/', $content, $matches)) {
        $metadata['addon_id'] = $matches[1];
    }
    
    return $metadata;
}

/**
 * Get language key from code (hr-HR -> hr)
 */
function getLanguageKey($code) {
    $parts = explode('-', $code);
    return strtolower($parts[0]);
}

// Scan language folders (format: xx-XX)
$dirs = glob($baseDir . '/*', GLOB_ONLYDIR);

foreach ($dirs as $dir) {
    $dirName = basename($dir);
    
    // Skip addons folder and hidden folders
    if ($dirName === 'addons' || $dirName[0] === '.') {
        continue;
    }
    
    // Check if it's a language folder (format: xx-XX or xx-xx, case insensitive)
    if (!preg_match('/^[a-z]{2}-[a-z]{2}$/i', $dirName)) {
        continue;
    }
    
    // Find XML files in this folder
    $xmlFiles = glob($dir . '/*.xml');
    
    foreach ($xmlFiles as $xmlFile) {
        $fileName = basename($xmlFile);
        
        // Process any XML file that looks like a language file
        if (preg_match('/-XF\d*\.xml$/i', $fileName) || stripos($fileName, $dirName) !== false || stripos($fileName, 'language') !== false) {
            $metadata = extractXmlMetadata($xmlFile);
            
            if ($metadata['code']) {
                $langKey = getLanguageKey($metadata['code']);
                
                $languages[$langKey] = [
                    'title' => $metadata['title'] ?: $dirName,
                    'code' => $metadata['code'],
                    'version' => $metadata['version'] ?: '1.0.0',
                    'file' => $fileName
                ];
                
                break; // Only one main language file per folder
            }
        }
    }
}

// Scan addons folder
$addonsDir = $baseDir . '/addons';
if (is_dir($addonsDir)) {
    $addonDirs = glob($addonsDir . '/*', GLOB_ONLYDIR);
    
    foreach ($addonDirs as $addonDir) {
        $folderName = basename($addonDir);
        $xmlFiles = glob($addonDir . '/*.xml');
        
        $translations = [];
        $detectedAddonId = null;
        
        foreach ($xmlFiles as $xmlFile) {
            $fileName = basename($xmlFile);
            $metadata = extractXmlMetadata($xmlFile);
            
            // Try to detect addon_id from XML file
            if ($metadata['addon_id'] && !$detectedAddonId) {
                $detectedAddonId = $metadata['addon_id'];
            }
            
            if ($metadata['code']) {
                $langKey = getLanguageKey($metadata['code']);
                
                $translations[$langKey] = [
                    'code' => $metadata['code'],
                    'version' => $metadata['version'] ?: '1.0.0',
                    'file' => $fileName
                ];
            }
        }
        
        if (!empty($translations)) {
            // Determine the actual addon_id
            // Priority: 1) Detected from XML, 2) Convert from folder name
            $actualAddonId = $detectedAddonId ?: folderToAddonId($folderName);
            
            $addonData = [
                'title' => $addonTitles[$folderName] ?? $folderName,
                'translations' => $translations
            ];
            
            // Only add addon_id if it's different from folder name
            // This helps Language Installer match the addon correctly
            if ($actualAddonId !== $folderName) {
                $addonData['addon_id'] = $actualAddonId;
            }
            
            $addons[$folderName] = $addonData;
        }
    }
}

// Sort languages and addons alphabetically
ksort($languages);
ksort($addons);

// Build final JSON structure
$output = [
    'languages' => $languages,
    'addons' => $addons,
    'version' => date('Y.m.d'),
    'updated' => date('Y-m-d H:i:s')
];

// Save to file
$jsonFile = $baseDir . '/languages.json';
$jsonContent = json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

if (file_put_contents($jsonFile, $jsonContent)) {
    header('Content-Type: application/json; charset=utf-8');
    echo $jsonContent;
    echo "\n\n// SUCCESS: languages.json has been generated!";
    echo "\n// Languages found: " . count($languages);
    echo "\n// Addons found: " . count($addons);
} else {
    die('ERROR: Could not write languages.json. Check folder permissions.');
}
