<?php

namespace DZ\LanguageInstaller\Admin\Controller;

use XF\Admin\Controller\AbstractController;
use XF\Mvc\ParameterBag;

class Language extends AbstractController
{
    /**
     * Repository base URL for language packs
     */
    protected $repoBaseUrl = 'https://dzmods.org/repo/language_installer';
    
    /**
     * Cache key for languages list
     */
    protected $cacheKey = 'dz_language_installer_list';
    
    /**
     * Cache lifetime in seconds (1 hour)
     */
    protected $cacheTtl = 3600;
    
    /**
     * Fallback languages if GitHub is unavailable
     */
    protected $fallbackLanguages = [
        'hr' => ['title' => 'Hrvatski (Croatian)', 'code' => 'hr-HR'],
        'ru' => ['title' => 'Русский (Russian)', 'code' => 'ru-RU'],
        'tr' => ['title' => 'Türkçe (Turkish)', 'code' => 'tr-TR'],
    ];
    
    /**
     * Get list of installed addon IDs
     */
    protected function getInstalledAddonIds()
    {
        $addons = \XF::app()->addOnManager()->getInstalledAddOns();
        return array_keys($addons);
    }
    
    /**
     * Get available languages from GitHub JSON or cache
     */
    protected function getAvailableLanguages()
    {
        $cache = \XF::app()->cache();
        
        // Try to get from cache first
        if ($cache)
        {
            $cached = $cache->fetch($this->cacheKey);
            if ($cached !== false)
            {
                return $cached;
            }
        }
        
        // Fetch from GitHub
        $languages = $this->fetchLanguagesFromGitHub();
        
        // Store in cache
        if ($cache && $languages)
        {
            $cache->save($this->cacheKey, $languages, $this->cacheTtl);
        }
        
        return $languages ?: $this->fallbackLanguages;
    }
    
    /**
     * Get repository metadata (version, updated date)
     */
    protected function getRepoMetadata()
    {
        $cache = \XF::app()->cache();
        $cacheKey = $this->cacheKey . '_meta';
        
        if ($cache)
        {
            $cached = $cache->fetch($cacheKey);
            if ($cached !== false)
            {
                return $cached;
            }
        }
        
        $metadata = $this->fetchRepoMetadata();
        
        if ($cache && $metadata)
        {
            $cache->save($cacheKey, $metadata, $this->cacheTtl);
        }
        
        return $metadata ?: ['version' => '-', 'updated' => '-'];
    }
    
    /**
     * Fetch repository metadata from JSON
     */
    protected function fetchRepoMetadata()
    {
        $url = $this->repoBaseUrl . '/languages.json';
        $client = $this->app()->http()->client();
        
        try
        {
            $response = $client->get($url, [
                'timeout' => 10,
                'allow_redirects' => true
            ]);
            
            if ($response->getStatusCode() === 200)
            {
                $json = json_decode($response->getBody()->getContents(), true);
                return [
                    'version' => $json['version'] ?? '-',
                    'updated' => $json['updated'] ?? '-'
                ];
            }
        }
        catch (\Exception $e)
        {
            // Silently fail
        }
        
        return null;
    }
    
    /**
     * Get available addon translations
     */
    protected function getAvailableAddonTranslations()
    {
        $cache = \XF::app()->cache();
        $cacheKey = $this->cacheKey . '_addons';
        
        if ($cache)
        {
            $cached = $cache->fetch($cacheKey);
            if ($cached !== false)
            {
                return $cached;
            }
        }
        
        $addons = $this->fetchAddonTranslationsFromGitHub();
        
        if ($cache && $addons)
        {
            $cache->save($cacheKey, $addons, $this->cacheTtl);
        }
        
        return $addons ?: [];
    }
    
    /**
     * Fetch languages.json from GitHub
     */
    protected function fetchLanguagesFromGitHub()
    {
        $url = $this->repoBaseUrl . '/languages.json';
        $client = $this->app()->http()->client();
        
        try
        {
            $response = $client->get($url, [
                'timeout' => 10,
                'allow_redirects' => true
            ]);
            
            if ($response->getStatusCode() === 200)
            {
                $json = json_decode($response->getBody()->getContents(), true);
                if (isset($json['languages']))
                {
                    return $json['languages'];
                }
            }
        }
        catch (\Exception $e)
        {
            \XF::logException($e, false, 'Language Installer: Failed to fetch languages from GitHub: ');
        }
        
        return null;
    }
    
    /**
     * Fetch addon translations from GitHub
     */
    protected function fetchAddonTranslationsFromGitHub()
    {
        $url = $this->repoBaseUrl . '/languages.json';
        $client = $this->app()->http()->client();
        
        try
        {
            $response = $client->get($url, [
                'timeout' => 10,
                'allow_redirects' => true
            ]);
            
            if ($response->getStatusCode() === 200)
            {
                $json = json_decode($response->getBody()->getContents(), true);
                if (isset($json['addons']))
                {
                    return $json['addons'];
                }
            }
        }
        catch (\Exception $e)
        {
            \XF::logException($e, false, 'Language Installer: Failed to fetch addon translations from GitHub: ');
        }
        
        return null;
    }
    
    /**
     * Clear the languages cache (useful after adding new languages)
     */
    public function actionRefresh()
    {
        $cache = \XF::app()->cache();
        if ($cache)
        {
            $cache->delete($this->cacheKey);
        }
        
        // Force refresh by fetching directly from GitHub (bypass cache)
        $languages = $this->fetchLanguagesFromGitHub();
        
        if ($languages)
        {
            // Store fresh data in cache
            if ($cache)
            {
                $cache->save($this->cacheKey, $languages, $this->cacheTtl);
            }
            
            return $this->redirect(
                $this->buildLink('language-installer'),
                \XF::phrase('li_language_list_refreshed') . ' (' . count($languages) . ' languages found)'
            );
        }
        
        return $this->redirect(
            $this->buildLink('language-installer'),
            \XF::phrase('li_could_not_fetch_language_list')
        );
    }
    
    /**
     * Get download URL for a language
     * Uses XenForo export format: language-{Title}-{AddonId}.xml
     * e.g., language-Hrvatski-(HR)-XF.xml
     */
    protected function getLanguageUrl($langData)
    {
        // If file is specified in JSON, use it directly
        if (isset($langData['file']))
        {
            return $this->repoBaseUrl . '/' . $langData['code'] . '/' . $langData['file'];
        }
        
        // Generate filename from title and code
        // Format: language-Hrvatski-(HR)-XF.xml
        $titlePart = $this->getTitleForFilename($langData['title'], $langData['code']);
        $filename = 'language-' . $titlePart . '-XF.xml';
        return $this->repoBaseUrl . '/' . $langData['code'] . '/' . $filename;
    }
    
    /**
     * Get download URL for addon translation
     * Uses XenForo export format: language-{Title}-{AddonId}.xml
     * e.g., language-Hrvatski-(HR)-XFRM.xml
     */
    protected function getAddonTranslationUrl($addonId, $translation, $langTitle)
    {
        // If file is specified in JSON, use it
        if (isset($translation['file']))
        {
            return $this->repoBaseUrl . '/addons/' . $addonId . '/' . $translation['file'];
        }
        
        // Generate filename: language-Hrvatski-(HR)-XFRM.xml
        $titlePart = $this->getTitleForFilename($langTitle, $translation['code']);
        $filename = 'language-' . $titlePart . '-' . $addonId . '.xml';
        return $this->repoBaseUrl . '/addons/' . $addonId . '/' . $filename;
    }
    
    /**
     * Generate title part for filename
     * Converts "Hrvatski (Croatian)" + "hr-HR" to "Hrvatski-(HR)"
     */
    protected function getTitleForFilename($title, $code)
    {
        // Extract country code from language code (hr-HR -> HR)
        $parts = explode('-', $code);
        $countryCode = isset($parts[1]) ? $parts[1] : strtoupper($parts[0]);
        
        // Remove anything in parentheses from title
        $cleanTitle = preg_replace('/\s*\([^)]+\)/', '', $title);
        $cleanTitle = trim($cleanTitle);
        
        // Build filename part: Hrvatski-(HR)
        return str_replace(' ', '-', $cleanTitle) . '-(' . $countryCode . ')';
    }

    /**
     * Get flag image URL from flagcdn.com
     */
    protected function getFlagUrl($countryCode)
    {
        $countryCode = strtolower($countryCode);
        return 'https://flagcdn.com/24x18/' . $countryCode . '.png';
    }
    
    /**
     * Extract country code from language code (e.g., hr-HR -> HR)
     */
    protected function getCountryFromLanguageCode($languageCode)
    {
        $parts = explode('-', $languageCode);
        return isset($parts[1]) ? $parts[1] : strtoupper($parts[0]);
    }
    
    /**
     * Convert XenForo version_id to readable version string
     * e.g., 2030770 -> 2.3.0, 2030870 -> 2.3.8
     */
    protected function versionIdToString($versionId)
    {
        if (!$versionId || !is_numeric($versionId))
        {
            return '-';
        }
        
        $versionId = (int)$versionId;
        $major = floor($versionId / 1000000);
        $minor = floor(($versionId % 1000000) / 10000);
        $patch = floor(($versionId % 10000) / 100);
        
        return $major . '.' . $minor . '.' . $patch;
    }

    public function actionIndex()
    {
        $existingLanguages = $this->finder('XF:Language')->fetch();
        $availableLanguages = $this->getAvailableLanguages();
        $existingCodes = [];
        $installedLanguages = [];
        
        // Process available languages to add readable version
        $processedLanguages = [];
        foreach ($availableLanguages as $key => $avail)
        {
            $processedLanguages[$key] = $avail;
            if (isset($avail['version_id']))
            {
                $processedLanguages[$key]['version'] = $this->versionIdToString($avail['version_id']);
            }
            elseif (!isset($avail['version']))
            {
                $processedLanguages[$key]['version'] = '-';
            }
        }
        
        foreach ($existingLanguages as $lang)
        {
            $countryCode = $this->getCountryFromLanguageCode($lang->language_code);
            $existingCodes[] = $lang->language_code;
            
            // Find version from available languages
            $version = '-';
            foreach ($processedLanguages as $avail)
            {
                if ($avail['code'] === $lang->language_code && isset($avail['version']))
                {
                    $version = $avail['version'];
                    break;
                }
            }
            
            $installedLanguages[] = [
                'language_id' => $lang->language_id,
                'title' => $lang->title,
                'language_code' => $lang->language_code,
                'flag_url' => $this->getFlagUrl($countryCode),
                'version' => $version
            ];
        }
        
        // Get addon translations
        $addonTranslations = $this->getAvailableAddonTranslations();
        
        // Get list of installed addons
        $installedAddons = $this->getInstalledAddonIds();
        
        // Build addon options for dropdown - only show if addon AND language is installed
        $addonOptions = [];
        foreach ($addonTranslations as $addonKey => $addon)
        {
            // Get actual addon ID - convert dash to slash if needed (MMO-Hide -> MMO/Hide)
            // Check if addon_id is explicitly set in JSON, otherwise try to map from key
            $actualAddonId = $addon['addon_id'] ?? str_replace('-', '/', $addonKey);
            
            // Also check if the key itself matches an installed addon (for simple IDs like XFMG)
            $isInstalled = in_array($actualAddonId, $installedAddons) || in_array($addonKey, $installedAddons);
            
            // Skip if addon is not installed
            if (!$isInstalled)
            {
                continue;
            }
            
            foreach ($addon['translations'] as $langKey => $translation)
            {
                // Only show translation if the language is installed
                if (in_array($translation['code'], $existingCodes))
                {
                    $optionKey = $addonKey . '_' . $langKey;
                    $addonOptions[$optionKey] = [
                        'addon_id' => $addonKey,
                        'addon_title' => $addon['title'],
                        'lang_key' => $langKey,
                        'code' => $translation['code'],
                        'version' => $translation['version'] ?? '-',
                        'file' => $translation['file'] ?? null,
                        'label' => $addon['title'] . ' - ' . $translation['code'] . ' (v' . ($translation['version'] ?? '?') . ')'
                    ];
                }
            }
        }
        
        // Get repository metadata
        $repoMetadata = $this->getRepoMetadata();

        $viewParams = [
            'languages' => $processedLanguages,
            'existingCodes' => $existingCodes,
            'installedLanguages' => $installedLanguages,
            'addonOptions' => $addonOptions,
            'repoVersion' => $repoMetadata['version'],
            'repoUpdated' => $repoMetadata['updated']
        ];

        return $this->view('DZ\LanguageInstaller:Language\Index', 'li_language_installer', $viewParams);
    }

    public function actionInstall()
    {
        $this->assertPostOnly();

        $languageKey = $this->filter('language', 'str');
        $availableLanguages = $this->getAvailableLanguages();

        if (!isset($availableLanguages[$languageKey]))
        {
            return $this->error(\XF::phrase('li_invalid_language_selected'));
        }

        $langData = $availableLanguages[$languageKey];
        $downloadUrl = $this->getLanguageUrl($langData);
        
        // Check if updating existing language
        $existingLanguage = $this->finder('XF:Language')
            ->where('language_code', $langData['code'])
            ->fetchOne();
        $isUpdate = (bool)$existingLanguage;

        try
        {
            // Download the language XML file
            $tempFile = $this->downloadLanguageFile($downloadUrl);

            if (!$tempFile)
            {
                return $this->error(\XF::phrase('li_could_not_download_language_file'));
            }

            // Import the language
            $this->importLanguageXml($tempFile, $langData['title']);

            // Clean up temp file
            @unlink($tempFile);

            $message = $isUpdate 
                ? \XF::phrase('li_language_updated_successfully', ['language' => $langData['title']])
                : \XF::phrase('li_language_installed_successfully', ['language' => $langData['title']]);

            return $this->redirect(
                $this->buildLink('language-installer'),
                $message
            );
        }
        catch (\Exception $e)
        {
            \XF::logException($e);
            return $this->error(\XF::phrase('li_error_installing_language') . ': ' . $e->getMessage());
        }
    }
    
    /**
     * Install addon translation into existing language
     */
    public function actionInstallAddon()
    {
        $this->assertPostOnly();

        $addonKey = $this->filter('addon_translation', 'str');
        
        if (empty($addonKey))
        {
            return $this->error(\XF::phrase('li_invalid_addon_selected') . ' (empty key)');
        }
        
        // Parse addon key (format: ADDONID_langkey)
        $parts = explode('_', $addonKey, 2);
        if (count($parts) !== 2)
        {
            return $this->error(\XF::phrase('li_invalid_addon_selected') . ' (bad format: ' . $addonKey . ')');
        }
        
        $addonId = $parts[0];
        $langKey = $parts[1];
        
        $addons = $this->getAvailableAddonTranslations();
        
        if (!isset($addons[$addonId]))
        {
            return $this->error('Addon not found in repository: ' . $addonId . '. Available: ' . implode(', ', array_keys($addons)));
        }
        
        if (!isset($addons[$addonId]['translations'][$langKey]))
        {
            return $this->error('Translation not found: ' . $langKey . ' for addon ' . $addonId);
        }
        
        $translation = $addons[$addonId]['translations'][$langKey];
        $addonTitle = $addons[$addonId]['title'];
        
        // Find the target language
        $targetLanguage = $this->finder('XF:Language')
            ->where('language_code', $translation['code'])
            ->fetchOne();
            
        if (!$targetLanguage)
        {
            return $this->error(\XF::phrase('li_target_language_not_installed', ['code' => $translation['code']]));
        }
        
        // Build download URL using XenForo format
        $downloadUrl = $this->getAddonTranslationUrl($addonId, $translation, $targetLanguage->title);

        try
        {
            $tempFile = $this->downloadLanguageFile($downloadUrl);

            if (!$tempFile)
            {
                return $this->error(\XF::phrase('li_could_not_download_language_file') . ' URL: ' . $downloadUrl);
            }

            // Import addon phrases into the target language
            $result = $this->importAddonPhrases($tempFile, $targetLanguage);

            @unlink($tempFile);

            // Redirect to rebuild caches for phrases to take effect
            $skippedInfo = '';
            if ($result['skipped'] > 0 && !empty($result['sampleSkipped'])) {
                $skippedInfo = '<br><br><strong>Skipped phrases (master not found):</strong> ' . implode(', ', $result['sampleSkipped']) . '...';
            }
            
            return $this->message(
                'Imported: ' . $result['imported'] . ' | Skipped: ' . $result['skipped'] . ' | Total in file: ' . $result['total'] . 
                '<br>Addon: ' . $addonTitle . ' | Language: ' . $targetLanguage->title .
                $skippedInfo .
                '<br><br><strong>If 0 imported:</strong> The addon may not be installed, or phrase names do not match.' .
                '<br><br><a href="' . $this->buildLink('tools/rebuild', null, ['job' => 'XF:Phrase']) . '" class="button">Rebuild Phrases</a>'
            );
        }
        catch (\Exception $e)
        {
            \XF::logException($e);
            return $this->error(\XF::phrase('li_error_installing_language') . ': ' . $e->getMessage());
        }
    }
    
    /**
     * Import addon phrases into existing language
     */
    protected function importAddonPhrases($file, $targetLanguage)
    {
        $xml = \XF\Util\Xml::openFile($file);

        if (!$xml)
        {
            throw new \InvalidArgumentException('Invalid XML file');
        }
        
        $phrasesImported = 0;
        $phrasesTotal = 0;
        $phrasesSkipped = 0;
        $sampleSkipped = [];
        
        // Check if it's a full language file or just phrases
        if ($xml->getName() === 'language')
        {
            // Full language file - XenForo format has <phrase> directly under <language>
            // Check for <phrases> wrapper first (some formats)
            if (isset($xml->phrases) && isset($xml->phrases->phrase))
            {
                foreach ($xml->phrases->phrase as $phraseXml)
                {
                    $phrasesTotal++;
                    if ($this->importSinglePhrase($phraseXml, $targetLanguage)) {
                        $phrasesImported++;
                    } else {
                        $phrasesSkipped++;
                        if (count($sampleSkipped) < 5) {
                            $sampleSkipped[] = (string)$phraseXml['title'];
                        }
                    }
                }
            }
            // Standard XenForo format - <phrase> directly under <language>
            elseif (isset($xml->phrase))
            {
                foreach ($xml->phrase as $phraseXml)
                {
                    $phrasesTotal++;
                    if ($this->importSinglePhrase($phraseXml, $targetLanguage)) {
                        $phrasesImported++;
                    } else {
                        $phrasesSkipped++;
                        if (count($sampleSkipped) < 5) {
                            $sampleSkipped[] = (string)$phraseXml['title'];
                        }
                    }
                }
            }
        }
        elseif ($xml->getName() === 'phrases')
        {
            // Just phrases wrapper
            foreach ($xml->phrase as $phraseXml)
            {
                $phrasesTotal++;
                if ($this->importSinglePhrase($phraseXml, $targetLanguage)) {
                    $phrasesImported++;
                } else {
                    $phrasesSkipped++;
                    if (count($sampleSkipped) < 5) {
                        $sampleSkipped[] = (string)$phraseXml['title'];
                    }
                }
            }
        }
        else
        {
            throw new \InvalidArgumentException('Invalid XML format: ' . $xml->getName());
        }
        
        if ($phrasesTotal === 0)
        {
            throw new \InvalidArgumentException('No phrases found in XML file');
        }
        
        // Compile phrases for this language
        $language = $this->em()->find('XF:Language', $targetLanguage->language_id);
        if ($language)
        {
            // Trigger recompile by touching the language
            $language->save();
        }

        return [
            'imported' => $phrasesImported, 
            'total' => $phrasesTotal,
            'skipped' => $phrasesSkipped,
            'sampleSkipped' => $sampleSkipped
        ];
    }
    
    /**
     * Import a single phrase into language
     */
    protected function importSinglePhrase($phraseXml, $targetLanguage)
    {
        $title = (string)$phraseXml['title'];
        $text = (string)$phraseXml;
        
        if (empty($title)) {
            return false;
        }
        
        // First check if master phrase exists (language_id = 0)
        // This confirms the phrase is defined by an addon
        $masterPhrase = $this->finder('XF:Phrase')
            ->where('language_id', 0)
            ->where('title', $title)
            ->fetchOne();
        
        if (!$masterPhrase)
        {
            // Master phrase doesn't exist - skip (addon not installed or phrase doesn't exist)
            return false;
        }
        
        // Check if translation already exists for this language
        $phrase = $this->finder('XF:Phrase')
            ->where('language_id', $targetLanguage->language_id)
            ->where('title', $title)
            ->fetchOne();
            
        if ($phrase)
        {
            // Update existing translation
            $phrase->phrase_text = $text;
            $phrase->save();
        }
        else
        {
            // Create translation for this language
            $phrase = $this->em()->create('XF:Phrase');
            $phrase->language_id = $targetLanguage->language_id;
            $phrase->title = $title;
            $phrase->phrase_text = $text;
            $phrase->addon_id = $masterPhrase->addon_id; // Use addon_id from master
            $phrase->save();
        }
        
        return true;
    }

    protected function downloadLanguageFile($url)
    {
        $client = $this->app()->http()->client();

        try
        {
            $response = $client->get($url, [
                'timeout' => 60,
                'allow_redirects' => true
            ]);

            if ($response->getStatusCode() !== 200)
            {
                return false;
            }

            $tempFile = \XF\Util\File::getTempFile();
            file_put_contents($tempFile, $response->getBody()->getContents());

            return $tempFile;
        }
        catch (\Exception $e)
        {
            \XF::logException($e);
            return false;
        }
    }

    protected function importLanguageXml($file, $title)
    {
        $xml = \XF\Util\Xml::openFile($file);

        if (!$xml || $xml->getName() !== 'language')
        {
            throw new \InvalidArgumentException('Invalid language XML file');
        }

        // Get language code from XML
        $languageCode = (string)$xml['language_code'];
        if (empty($languageCode))
        {
            $languageCode = (string)$xml['code'];
        }

        $existingLanguage = null;
        if (!empty($languageCode))
        {
            $existingLanguage = $this->finder('XF:Language')
                ->where('language_code', $languageCode)
                ->fetchOne();
        }

        /** @var \XF\Service\Language\Import $importService */
        $importService = $this->service('XF:Language\Import');

        if ($existingLanguage && $existingLanguage->language_id > 1)
        {
            // Delete existing language first, then reimport
            $parentId = $existingLanguage->parent_id ?: 0;
            
            // Delete the existing language
            $existingLanguage->delete();
            
            // Import as new language with same parent
            $importService->importFromXml($xml, $parentId);
        }
        else
        {
            // Import as new language with parent = 0 (English base)
            $importService->importFromXml($xml, 0);
        }

        return true;
    }

    public function actionCustomUrl()
    {
        if ($this->isPost())
        {
            $url = $this->filter('custom_url', 'str');
            $languageTitle = $this->filter('language_title', 'str');

            if (empty($url) || empty($languageTitle))
            {
                return $this->error(\XF::phrase('li_please_provide_url_and_title'));
            }

            try
            {
                $tempFile = $this->downloadLanguageFile($url);

                if (!$tempFile)
                {
                    return $this->error(\XF::phrase('li_could_not_download_language_file'));
                }

                $this->importLanguageXml($tempFile, $languageTitle);

                @unlink($tempFile);

                return $this->redirect(
                    $this->buildLink('language-installer'),
                    \XF::phrase('li_language_installed_successfully', ['language' => $languageTitle])
                );
            }
            catch (\Exception $e)
            {
                \XF::logException($e);
                return $this->error(\XF::phrase('li_error_installing_language') . ': ' . $e->getMessage());
            }
        }

        return $this->view('DZ\LanguageInstaller:Language\CustomUrl', 'li_custom_url', []);
    }

    public function actionUninstall()
    {
        $languageId = $this->filter('language_id', 'uint');
        
        $language = $this->assertLanguageExists($languageId);
        
        // Don't allow deleting the master language
        if ($languageId <= 1)
        {
            return $this->error(\XF::phrase('li_cannot_delete_base_language'));
        }

        if ($this->isPost())
        {
            $language->delete();
            
            return $this->redirect(
                $this->buildLink('language-installer'),
                \XF::phrase('li_language_uninstalled_successfully', ['language' => $language->title])
            );
        }
        else
        {
            $viewParams = [
                'language' => $language
            ];
            return $this->view('DZ\LanguageInstaller:Language\Uninstall', 'li_uninstall_confirm', $viewParams);
        }
    }

    protected function assertLanguageExists($id)
    {
        $language = $this->finder('XF:Language')->where('language_id', $id)->fetchOne();
        if (!$language)
        {
            throw $this->exception($this->notFound(\XF::phrase('requested_language_not_found')));
        }
        return $language;
    }
}
