<?php

namespace DZ\AddonInstaller\Admin\Controller;

use XF\Admin\Controller\AbstractController;
use XF\Mvc\ParameterBag;

class Addon extends AbstractController
{
    /**
     * Repository base URL for addons
     */
    protected $repoBaseUrl = 'https://dzmods.org/repo/addon_installer';
    
    /**
     * Cache key for addons list
     */
    protected $cacheKey = 'dz_addon_installer_list';
    
    /**
     * Cache lifetime in seconds (1 hour)
     */
    protected $cacheTtl = 3600;
    
    /**
     * Get available addons from repository JSON or cache
     */
    protected function getAvailableAddons()
    {
        $cache = \XF::app()->cache();
        
        if ($cache)
        {
            $cached = $cache->fetch($this->cacheKey);
            if ($cached !== false)
            {
                return $cached;
            }
        }
        
        $addons = $this->fetchAddonsFromRepo();
        
        if ($cache && $addons)
        {
            $cache->save($this->cacheKey, $addons, $this->cacheTtl);
        }
        
        return $addons ?: [];
    }
    
    /**
     * Get repository metadata (version, updated date)
     */
    protected function getRepoMetadata()
    {
        $cache = \XF::app()->cache();
        $cacheKey = $this->cacheKey . '_meta';
        
        if ($cache)
        {
            $cached = $cache->fetch($cacheKey);
            if ($cached !== false)
            {
                return $cached;
            }
        }
        
        $metadata = $this->fetchRepoMetadata();
        
        if ($cache && $metadata)
        {
            $cache->save($cacheKey, $metadata, $this->cacheTtl);
        }
        
        return $metadata ?: ['version' => '-', 'updated' => '-'];
    }
    
    /**
     * Fetch repository metadata from JSON
     */
    protected function fetchRepoMetadata()
    {
        $url = $this->repoBaseUrl . '/addons.json';
        $client = $this->app()->http()->client();
        
        try
        {
            $response = $client->get($url, [
                'timeout' => 10,
                'allow_redirects' => true
            ]);
            
            if ($response->getStatusCode() === 200)
            {
                $json = json_decode($response->getBody()->getContents(), true);
                return [
                    'version' => $json['version'] ?? '-',
                    'updated' => $json['updated'] ?? '-'
                ];
            }
        }
        catch (\Exception $e)
        {
            // Silently fail
        }
        
        return null;
    }
    
    /**
     * Fetch addons.json from repository
     */
    protected function fetchAddonsFromRepo()
    {
        $url = $this->repoBaseUrl . '/addons.json';
        $client = $this->app()->http()->client();
        
        try
        {
            $response = $client->get($url, [
                'timeout' => 10,
                'allow_redirects' => true
            ]);
            
            if ($response->getStatusCode() === 200)
            {
                $json = json_decode($response->getBody()->getContents(), true);
                if (isset($json['addons']))
                {
                    return $json['addons'];
                }
            }
        }
        catch (\Exception $e)
        {
            \XF::logException($e, false, 'Addon Installer: Failed to fetch addons from repository: ');
        }
        
        return null;
    }
    
    /**
     * Get list of installed addon IDs with their versions
     */
    protected function getInstalledAddons()
    {
        $addons = \XF::app()->addOnManager()->getInstalledAddOns();
        $installed = [];
        
        foreach ($addons as $addonId => $addon)
        {
            $installed[$addonId] = [
                'title' => $addon->title,
                'version_string' => $addon->version_string,
                'version_id' => $addon->version_id
            ];
        }
        
        return $installed;
    }
    
    /**
     * Convert version_id to readable version string
     */
    protected function versionIdToString($versionId)
    {
        if (!$versionId || !is_numeric($versionId))
        {
            return '-';
        }
        
        $versionId = (int)$versionId;
        $major = floor($versionId / 1000000);
        $minor = floor(($versionId % 1000000) / 10000);
        $patch = floor(($versionId % 10000) / 100);
        
        return $major . '.' . $minor . '.' . $patch;
    }
    
    /**
     * Clear the addons cache
     */
    public function actionRefresh()
    {
        $cache = \XF::app()->cache();
        if ($cache)
        {
            $cache->delete($this->cacheKey);
            $cache->delete($this->cacheKey . '_meta');
        }
        
        $addons = $this->fetchAddonsFromRepo();
        
        if ($addons)
        {
            if ($cache)
            {
                $cache->save($this->cacheKey, $addons, $this->cacheTtl);
            }
            
            return $this->redirect(
                $this->buildLink('addon-installer'),
                \XF::phrase('ai_addon_list_refreshed') . ' (' . count($addons) . ' addons found)'
            );
        }
        
        return $this->redirect(
            $this->buildLink('addon-installer'),
            \XF::phrase('ai_could_not_fetch_addon_list')
        );
    }

    public function actionIndex()
    {
        $availableAddons = $this->getAvailableAddons();
        $installedAddons = $this->getInstalledAddons();
        $repoMetadata = $this->getRepoMetadata();
        
        // Process addons to add install/update status
        $processedAddons = [];
        foreach ($availableAddons as $addonId => $addon)
        {
            $isInstalled = isset($installedAddons[$addonId]);
            $installedVersion = $isInstalled ? $installedAddons[$addonId]['version_id'] : 0;
            $availableVersion = $addon['version_id'] ?? 0;
            $hasUpdate = $isInstalled && ($availableVersion > $installedVersion);
            
            // Determine icon class
            $icon = $addon['icon'] ?? '';
            $iconClass = 'fa fa-cube'; // default
            if ($icon)
            {
                // Check if it's a FontAwesome icon (no file extension)
                if (strpos($icon, '.') === false)
                {
                    // It's a FA icon like "fa-users" or "fad fa-joystick"
                    if (preg_match('/^(fas|far|fal|fad|fab)\s/', $icon))
                    {
                        // Already has style prefix like "fad fa-joystick"
                        $iconClass = $icon;
                    }
                    else
                    {
                        // Just icon name like "fa-users", add "fa " prefix
                        $iconClass = 'fa ' . $icon;
                    }
                }
                else
                {
                    // It's an image file, use placeholder
                    $iconClass = 'fa fa-puzzle-piece';
                }
            }
            
            $processedAddons[$addonId] = $addon;
            $processedAddons[$addonId]['is_installed'] = $isInstalled;
            $processedAddons[$addonId]['has_update'] = $hasUpdate;
            $processedAddons[$addonId]['installed_version'] = $isInstalled ? $installedAddons[$addonId]['version_string'] : null;
            $processedAddons[$addonId]['icon_class'] = $iconClass;
        }
        
        // Pagination
        $page = $this->filterPage();
        $perPage = 20;
        $total = count($processedAddons);
        
        // Slice addons for current page
        $offset = ($page - 1) * $perPage;
        $pagedAddons = array_slice($processedAddons, $offset, $perPage, true);

        $viewParams = [
            'addons' => $processedAddons,
            'pagedAddons' => $pagedAddons,
            'installedAddons' => $installedAddons,
            'repoVersion' => $repoMetadata['version'],
            'repoUpdated' => $repoMetadata['updated'],
            'page' => $page,
            'perPage' => $perPage,
            'total' => $total
        ];

        return $this->view('DZ\AddonInstaller:Addon\Index', 'ai_addon_installer', $viewParams);
    }

    public function actionInstall()
    {
        $this->assertPostOnly();

        $addonId = $this->filter('addon', 'str');
        $availableAddons = $this->getAvailableAddons();

        if (!isset($availableAddons[$addonId]))
        {
            return $this->error(\XF::phrase('ai_invalid_addon_selected'));
        }

        $addonData = $availableAddons[$addonId];
        
        // Check if addon is already installed
        $installedAddons = $this->getInstalledAddons();
        $isUpdate = isset($installedAddons[$addonId]);

        try
        {
            // Download the ZIP file - files are directly in repo root
            $zipUrl = $this->repoBaseUrl . '/' . $addonData['file'];
            $zipContent = $this->downloadFile($zipUrl);
            
            if (!$zipContent)
            {
                return $this->error(\XF::phrase('ai_could_not_download_addon'));
            }
            
            // Save to temp file
            $tempFile = \XF\Util\File::getTempDir() . '/addon_' . md5($addonId) . '.zip';
            file_put_contents($tempFile, $zipContent);
            
            // Extract ZIP to addons directory
            $addonsPath = \XF::getAddOnDirectory();
            
            $zip = new \ZipArchive();
            if ($zip->open($tempFile) !== true)
            {
                @unlink($tempFile);
                return $this->error(\XF::phrase('ai_could_not_extract_addon'));
            }
            
            // Find the root directory in ZIP (e.g., "upload/" or addon folder)
            $rootDir = '';
            for ($i = 0; $i < $zip->numFiles; $i++)
            {
                $name = $zip->getNameIndex($i);
                if (preg_match('#^(upload/)?#', $name, $matches))
                {
                    if (!empty($matches[1]))
                    {
                        $rootDir = 'upload/';
                    }
                    break;
                }
            }
            
            // Extract files
            for ($i = 0; $i < $zip->numFiles; $i++)
            {
                $name = $zip->getNameIndex($i);
                
                // Skip if not in our expected path
                if ($rootDir && strpos($name, $rootDir) !== 0)
                {
                    continue;
                }
                
                // Remove root dir prefix
                $relativePath = $rootDir ? substr($name, strlen($rootDir)) : $name;
                
                // We want files from src/addons/
                if (strpos($relativePath, 'src/addons/') === 0)
                {
                    $targetPath = $addonsPath . '/' . substr($relativePath, strlen('src/addons/'));
                }
                else
                {
                    continue;
                }
                
                // Create directories
                if (substr($name, -1) === '/')
                {
                    if (!is_dir($targetPath))
                    {
                        \XF\Util\File::createDirectory($targetPath, true);
                    }
                }
                else
                {
                    // Extract file
                    $dir = dirname($targetPath);
                    if (!is_dir($dir))
                    {
                        \XF\Util\File::createDirectory($dir, true);
                    }
                    
                    $content = $zip->getFromIndex($i);
                    file_put_contents($targetPath, $content);
                }
            }
            
            $zip->close();
            @unlink($tempFile);
            
            // Now install or upgrade the addon
            $addOnManager = \XF::app()->addOnManager();
            $addOn = $addOnManager->getById($addonId);
            
            if (!$addOn)
            {
                return $this->error(\XF::phrase('ai_installation_failed') . ': Add-on files not found after extraction');
            }
            
            // Files extracted - redirect to add-ons page with message
            $message = \XF::phrase('ai_addon_files_extracted');
            
            if ($isUpdate)
            {
                // For upgrades, go directly to upgrade page
                return $this->redirect($this->buildLink('add-ons/upgrade', $addOn));
            }
            else
            {
                // For fresh installs, redirect to add-ons list where they can see it
                return $this->redirect(
                    $this->buildLink('add-ons'),
                    $message . ' - ' . \XF::phrase('ai_click_install_to_complete', ['title' => $addonData['title']])
                );
            }
        }
        catch (\Exception $e)
        {
            \XF::logException($e, false, 'Addon Installer: ');
            return $this->error(\XF::phrase('ai_installation_failed') . ': ' . $e->getMessage());
        }
    }
    
    /**
     * Download file from URL
     */
    protected function downloadFile($url)
    {
        $client = $this->app()->http()->client();
        
        try
        {
            $response = $client->get($url, [
                'timeout' => 60,
                'allow_redirects' => true
            ]);
            
            if ($response->getStatusCode() === 200)
            {
                return $response->getBody()->getContents();
            }
        }
        catch (\Exception $e)
        {
            \XF::logException($e, false, 'Addon Installer: Failed to download: ');
        }
        
        return null;
    }
    
    /**
     * Extract ZIP file to destination
     */
    protected function extractZip($zipFile, $destination)
    {
        $zip = new \ZipArchive();
        
        if ($zip->open($zipFile) === true)
        {
            $zip->extractTo($destination);
            $zip->close();
            return true;
        }
        
        return false;
    }
    
    /**
     * View addon details
     */
    public function actionDetails(ParameterBag $params)
    {
        $addonId = $this->filter('addon_id', 'str');
        $availableAddons = $this->getAvailableAddons();
        
        if (!isset($availableAddons[$addonId]))
        {
            return $this->error(\XF::phrase('ai_addon_not_found'));
        }
        
        $addon = $availableAddons[$addonId];
        $installedAddons = $this->getInstalledAddons();
        $isInstalled = isset($installedAddons[$addonId]);
        
        $viewParams = [
            'addon' => $addon,
            'addonId' => $addonId,
            'isInstalled' => $isInstalled,
            'installedVersion' => $isInstalled ? $installedAddons[$addonId]['version_string'] : null
        ];
        
        return $this->view('DZ\AddonInstaller:Addon\Details', 'ai_addon_details', $viewParams);
    }
}
